<?php
session_start();
require_once __DIR__.'/app/functions.php';
require_login();

$uid = current_user_id();
$username = $_SESSION['user']['username'] ?? 'Unknown';
$role = $_SESSION['user']['role'] ?? 'user';

// DB connection
$pdo = db();

// Date filter
$from = $_GET['from'] ?? date('Y-m-01');
$to = $_GET['to'] ?? date('Y-m-d');

// Fetch locations
$locations = $pdo->query("SELECT id, name FROM locations ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

// Build base query
$sql = "SELECT c.*, u.username, l.name AS location_name 
        FROM couriers c
        JOIN users u ON c.created_by=u.id
        JOIN locations l ON c.sending_from=l.id
        WHERE c.created_at BETWEEN :from AND :to";

$params = [':from'=>$from.' 00:00:00', ':to'=>$to.' 23:59:59'];

if($role!=='admin'){
    $sql .= " AND c.created_by=:uid";
    $params[':uid']=$uid;
}

$sql .= " ORDER BY l.name, u.username, c.created_at";

$couriers = $pdo->prepare($sql);
$couriers->execute($params);
$couriers = $couriers->fetchAll(PDO::FETCH_ASSOC);

// Group data
$report = [];
$locationTotals = [];
foreach($couriers as $c){
    $loc = $c['location_name'];
    $user = $c['username'];
    if(!isset($report[$loc])) $report[$loc]=[];
    if(!isset($report[$loc][$user])) $report[$loc][$user]=['couriers'=>[], 'subtotal'=>0];

    $report[$loc][$user]['couriers'][] = $c;
    $report[$loc][$user]['subtotal'] += $c['amount'];
    $locationTotals[$loc] = ($locationTotals[$loc] ?? 0) + $c['amount'];
}

$grandTotal = array_sum($locationTotals);
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Courier Report</title>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css">
<style>
body { font-size:14px; background:#f8f9fa; }
.cursor-pointer { cursor:pointer; }
header { background:#008c8c; padding:8px; color:#fff; margin-bottom:10px; }
header a { color:#fff; text-decoration:none; }
@media print {
    body * { visibility: hidden !important; }
    #reportContainer, #reportContainer * { visibility: visible !important; }
    #reportContainer { position: absolute; left:0; top:0; width:100%; }
}
</style>
</head>
<body class="p-3">

<?php include("header.php"); ?>

<div class="container-fluid" id="reportContainer">
    <h4 class="mb-3">Courier Report</h4>

    <!-- Filter Form -->
    <form class="row g-2 mb-3" method="get">
        <div class="col-auto">
            <label>From:</label>
            <input type="date" name="from" value="<?=htmlspecialchars($from)?>" class="form-control form-control-sm">
        </div>
        <div class="col-auto">
            <label>To:</label>
            <input type="date" name="to" value="<?=htmlspecialchars($to)?>" class="form-control form-control-sm">
        </div>
        <div class="col-auto align-self-end">
            <button type="submit" class="btn btn-primary btn-sm">Filter</button>
        </div>
        <div class="col-auto align-self-end">
            <button type="button" id="printBtn" class="btn btn-success btn-sm">🖨 Print</button>
            <button type="button" id="exportExcel" class="btn btn-outline-secondary btn-sm">📥 Export Excel</button>
            <button type="button" id="exportPDF" class="btn btn-outline-secondary btn-sm">📥 Export PDF</button>
        </div>
    </form>

    <!-- Report -->
    <?php foreach($report as $location => $users): ?>
        <div class="card mb-2">
            <div class="card-header bg-info text-white cursor-pointer" data-bs-toggle="collapse" data-bs-target="#loc-<?=md5($location)?>">
                <strong>Location:</strong> <?=htmlspecialchars($location)?> 
            </div>
            <div id="loc-<?=md5($location)?>" class="collapse show">
                <div class="card-body p-1">
                <?php foreach($users as $user => $data): ?>
                    <div class="card mb-1">
                        <div class="card-header bg-light cursor-pointer" data-bs-toggle="collapse" data-bs-target="#user-<?=md5($location.$user)?>">
                            User: <?=htmlspecialchars($user)?>
                        </div>
                        <div id="user-<?=md5($location.$user)?>" class="collapse show">
                            <div class="table-responsive">
                                <table class="table table-sm table-bordered mb-0">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Sender</th>
                                            <th>Recipient</th>
                                            <th>Price</th>
                                            <th>Qty</th>
                                            <th>Amount</th>
                                            <th>Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach($data['couriers'] as $c): ?>
                                        <tr>
                                            <td><?=$c['id']?></td>
                                            <td><?=$c['sender_name']?></td>
                                            <td><?=$c['recipient_name']?></td>
                                            <td><?=$c['price']?></td>
                                            <td><?=$c['quantity']?></td>
                                            <td><?=$c['amount']?></td>
                                            <td><?=$c['created_at']?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                        <tr class="table-secondary">
                                            <td colspan="5" class="text-end"><strong>Subtotal:</strong></td>
                                            <td colspan="2"><strong><?=number_format($data['subtotal'],2)?></strong></td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
                <div class="table-responsive mt-1">
                    <table class="table table-sm table-bordered mb-0">
                        <tr class="table-warning">
                            <td colspan="5" class="text-end"><strong>Location Total:</strong></td>
                            <td colspan="2"><strong><?=number_format($locationTotals[$location],2)?></strong></td>
                        </tr>
                    </table>
                </div>
                </div>
            </div>
        </div>
    <?php endforeach; ?>

    <div class="card mt-3">
        <div class="card-body bg-warning text-dark">
            <h5 class="mb-0 text-center"><strong>Grand Total: <?=number_format($grandTotal,2)?></strong></h5>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.19.2/xlsx.full.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
<script>
// Print
document.getElementById('printBtn').onclick = () => window.print();

// Export Excel
document.getElementById('exportExcel').onclick = () => {
    let wb = XLSX.utils.book_new();
    document.querySelectorAll('#reportContainer table').forEach((tbl, idx) => {
        let ws = XLSX.utils.table_to_sheet(tbl);
        XLSX.utils.book_append_sheet(wb, ws, 'Sheet' + (idx+1));
    });
    XLSX.writeFile(wb, 'courier_report.xlsx');
};

// Export PDF
document.getElementById('exportPDF').onclick = () => {
    const { jsPDF } = window.jspdf;
    let doc = new jsPDF('p', 'pt', 'a4');
    doc.html(document.querySelector('#reportContainer'), {
        callback: function(pdf){ pdf.save('courier_report.pdf'); },
        x: 10, y: 10,
        html2canvas: { scale: 0.8 }
    });
};
</script>
</body>
</html>
